/*
    geniedat - A library for reading and writing data files of genie
               engine games.
    Copyright (C) 2011 - 2013  Armin Preiml
    Copyright (C) 2011 - 2021  Mikko "Tapsa" P

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef GENIE_UNIT_H
#define GENIE_UNIT_H
#include "genie/file/ISerializable.h"
#include "unit/DamageGraphic.h"
#include "UnitCommand.h"
#include "unit/AttackOrArmor.h"
#include "unit/TrainLocation.h"
#include "unit/DeadFish.h"
#include "unit/Bird.h"
#include "unit/Type50.h"
#include "unit/Projectile.h"
#include "unit/Creatable.h"
#include "unit/Building.h"
#include "ResourceUsage.h"

namespace genie
{

/// Available unit types
enum UnitType
{
  /// Basic units like rubble and flares.
  UT_EyeCandy = 10,

  /// Trees, used to be 90?
  UT_Trees = 15,

  /// With Speed but mostly flags.
  UT_Flag = 20,

  /// Only one unit has this type. AOK, DOPL (id 243) same properties as UT_Flag
  UT_25 = 25,

  /// Dead and fish units. It seems to be unused in SWGB as units just explode
  /// and do not leave carcasses.
  UT_Dead_Fish = 30,

  /// Only birds in aoe and ror are of this type.
  UT_Bird = 40,

  /// Shared class inherited by combat objects.
  UT_Combatant = 50,

  /// Projectiles
  UT_Projectile = 60,

  /// Units that can be created or trained like Army, Villagers and Ships.
  UT_Creatable = 70,

  /// Buildings
  UT_Building = 80,

  /// Trees in aoe and ror are of this type
  UT_AoeTrees = 90
};

//------------------------------------------------------------------------------
/// Stores properties of genie units.
//
class Unit : public ISerializable
{
public:
  Unit();
  virtual ~Unit();
  virtual void setGameVersion(GameVersion gv);

  /// See enum UnitType
  uint8_t Type = 10;

  int16_t ID = -1;

  union
  {
     /// ID of the name string stored in the language.dll file
     int32_t LanguageDLLName = 5000;
     int16_t LanguageDLLNameU16;
  };
  union
  {
     /// ID of the name string stored in the language.dll file
     int32_t LanguageDLLCreation = 6000;
     int16_t LanguageDLLCreationU16;
  };

  /// Class that gives certain properties
  int16_t Class = -1;

  /// Default standing graphic
  std::pair<int16_t, int16_t> StandingGraphic = {-1, -1};

  /// Graphic shown while the units is dying.
  int16_t DyingGraphic = -1;
  int16_t UndeadGraphic = -1;

  /// Enables undead state
  uint8_t UndeadMode = 0;

  /// Unit's hit points
  int16_t HitPoints = 1;

  /// Default line of sight measured in tiles
  float LineOfSight = 2;

  /// Maximum number of units that can be garrisonned at the same time.
  uint8_t GarrisonCapacity = 0;

  /// Collision detection area taken by the unit.
  /// No other unit can move into this area except flying units.
  XYZF CollisionSize = {0, 0, 0};

  /// Sound played when the unit is created
  int16_t TrainSound = -1;
  int16_t DamageSound = -1;
  uint32_t WwiseTrainSoundID = 0;
  uint32_t WwiseDamageSoundID = 0;

  /// ID of the unit to change to when this one dies.
  int16_t DeadUnitID = -1;
  int16_t BloodUnitID = -1;

  uint8_t SortNumber = 0;

  uint8_t CanBeBuiltOn = 0;

  /// ID of the icon shown at the bottom when selected or when building
  int16_t IconID = -1;

  /// If true, unit can not be placed in the editor
  uint8_t HideInEditor = 0;

  int16_t OldPortraitPict = -1;         // Always -1

  uint8_t Enabled = 1;
  uint8_t Disabled = 0;
  std::pair<int16_t, int16_t> PlacementSideTerrain = {-1, -1};
  std::pair<int16_t, int16_t> PlacementTerrain = {-1, -1};
  std::pair<float, float> ClearanceSize = {0.f, 0.f};
  uint8_t HillMode = 0;
  uint8_t FogVisibility = 0;

  /// ID of terrain restrictions that are imposed on the unit.
  int16_t TerrainRestriction = 0;

  /// Bool which determines wheter the unit can fly.
  uint8_t FlyMode = 0;

  /// How much resources this unit is able to carry
  int16_t ResourceCapacity = 0;

  float ResourceDecay = 0;
  uint8_t BlastDefenseLevel = 0;
  uint8_t CombatLevel = 0;
  uint8_t InteractionMode = 0;

  /**
   * \brief Defines if and how a unit is shown on the minimap.
   *
   * Following values are known to work:
   *
   * - 0 does not place a dot on the mini-map. Eye candy, resources,
   *     fish traps, OUTLWs, and farms have this value
   * - 1 places a square dot on the min-map for the unit. When the unit is
   *     selected, the dot will turn white
   * - 2 places a diamond-shaped dot on the mini-map for the unit. When the unit
   *     is selected, the dot will turn white
   * - 3 places a diamond-shaped dot on the mini-map for the unit. When the unit
   *     is selected, the dot remains the player's color.
   * - 4 places a larger black spot on the minimap which does not follow the
   *     unit. Whenever you open the chat box (enter) and close it, the black
   *     spots will move to the unit on the minimap. No blinking occurs when
   *     the unit is attacked. Both allied and enemy units are visible in the
   *     black area on the minimap. CLF units have this value and a brown spot
   *     is placed instead.
   * - 5 does the same thing as “4”
   * - 6 does not place a dot on the minimap
   * - 7-9 does the same thing as “6”
   * - 10 does the same thing as “6”. Hawks, macaws, and flying dogs have
   *      this value.
   */
  uint8_t MinimapMode = 0;

  uint8_t InterfaceKind = 0;// Page for Build button: 2 = Page 1, 10 = Page 2, ?11 = Page 3?
  float MultipleAttributeMode = 0;
  uint8_t MinimapColor = 0;
  int32_t LanguageDLLHelp = 105000;
  int32_t LanguageDLLHotKeyText = 155000;
  uint8_t Recyclable = 0;
  uint8_t EnableAutoGather = 0;
  uint8_t CreateDoppelgangerOnDeath = 0;
  uint8_t ResourceGatherGroup = 0;
  uint8_t OcclusionMode = 0;

  /// values for ObstructionType
  /// 0 farm, gate, dead bodies, town center
  /// 2 buildings, gold mine
  /// 3 berserk, flag x
  /// 5 units
  /// 10 mountain(matches selction mask)
  uint8_t ObstructionType = 0;

  /// 0 default, 1+ above
  uint8_t ObstructionClass = 0;
  uint8_t Trait = 0;
  /// Seems to be used only in SWGB/CC
  uint8_t Civilization = 0;
  int16_t Nothing = 0;

  /**
   * 1.HP Bar on, selection shadow (permanent darker outline in editor only, disappears in game)
   * 2.HP Bar on, normal outline
   * 3.HP Bar off, selection shadow
   * 4.HP Bar off, normal outline
   * Note: this doe not work with all units
   */
  uint8_t SelectionEffect = 1;
  uint8_t EditorSelectionColour = 52;
  XYZF OutlineSize = {0, 0, 0};

  typedef ResourceUsage<int16_t, float, uint8_t> ResourceStorage;

  /// Resource cost of a unit
  std::vector<ResourceStorage> ResourceStorages;

  std::vector<unit::DamageGraphic> DamageGraphics;

  /// Sound that is played when this unit is selected
  int16_t SelectionSound = -1;
  int16_t DyingSound = -1;
  uint32_t WwiseSelectionSoundID = 0;
  uint32_t WwiseDyingSoundID = 0;
  uint8_t OldAttackReaction = 0;
  uint8_t ConvertTerrain = 0;
  std::string Name = "";
  std::string Name2 = "";

  /// MinGameVersion: SWGB
  int16_t Unitline = -1;

  /// MinGameVersion: SWGB
  uint8_t MinTechLevel = -1;

  int16_t CopyID = -1;
  int16_t BaseID = -1;//not in aoe/ror
  int16_t TelemetryID = -1;

//      Type 20+

  float Speed = 0;

  unit::DeadFish DeadFish;

  unit::Bird Bird;

  unit::Type50 Type50;

  unit::Projectile Projectile;

  unit::Creatable Creatable;

  unit::Building Building;

protected:
  virtual void serializeObject(void);
};

}

#endif // GENIE_UNIT_H
