{$A+,B-,D+,E-,F-,I+,L-,N-,O-,R-,S+,V-}
{$M 16384,0,131072}

program Animals;

{
  DESCRIPTION
    This program is a simple example of artificial intelligence.  It
    is an animal guessing game, where the user thinks of an animal and
    the program does the "guessing."  If the program does not know the
    the animal, it asks for a description and thus "learns" from the
    user.

  EFFICIENCY
    If the early questions are carefully structured, the program can achieve
    remarkable efficiency.  This program can be easily adapted for use in
    other applications.  It should also be noted that the program will lose
    badly if the user lies or is ignorant about the animal in question (i.e.
    that horses are single-toed, hoofed herbivores).

  PROGRAM OVERVIEW
    Each animal and its brief description are stored in a record (AnimalRec).
    The record also contains two pointers that are used to build a binary
    tree.  Beginning with the first animal in the tree (the root), the
    descriptive statement is phrased in the form of a yes or no question
    (It is a bird?).  If the answer is no, then the guessing procedure
    (GuessAnimal) is called recursively and passed the value of the NO
    pointer.  If the answer is yes, then the program "guesses" the animal
    associated with the question:

                It's cold blooded w/scaly skin
                (snake)
                YES                 NO
                /                     \
         Likes water              It' a bird
         (frog)                   (sparrow)
         YES     NO               YES          NO
         /         \              /               \
 Sharp teeth     Pushups      Flightless         Flies
   ...             ...          ...                ...

  If the program guesses your animal, it brags for a moment and then
  offers you a chance to play again (function WantToGuess).  If the program
  does not know your animal (it gets to a dead end), it will ask you for
  the animal's name and a descriptive sentence, add these to its tree
  and then start over.

  This aspect of the program is actually quite short.  In addition to
  behaving as described above, this program saves its data to disk in a
  text file (ANIMALS.DTA) and loads the tree automatically the next time
  the program is run.  It also features some modest cartooning (an owl that
  blinks its eyes) to make it more fun to play.
  -------------------------------------------------------------------------
}

Uses
  Crt;

const
  DataFile : string[11]='ANIMALS.DTA';             { disk file to store tree }
  MaxRows :  byte=24;                                  { max rows per screen }
  ESC  = #27;                             { several keystrokes' ASCII codes: }
  CR   = #13;
  NULL = #0;
  BS   = #8;
  DEL  = #127;

type
  MaxString = string[80];                 { general purpose string type }
  NameStr = string[25];                   { animal's name }
  QuestionStr = string[40];               { animal's description }
  AnimalPtr = ^AnimalRec;
  AnimalRec = record
                Name : NameStr;
                Question : QuestionStr;
                Yes,
                No : AnimalPtr;
              end;
  StringRec = record                       { used to load tree from disk }
                Name : NameStr;
                Question : QuestionStr;
                StringP : string[3];      { NIL or END }
              end;

  ScoreRec = record
               Added,                     { number of new animals learned }
               Guessed,                   { number of correct guesses     }
               Total : integer;           { total animals known           }
             end;
  BubbleRec = record
                X, Y,                     { box's dimensions }
                Height,
                Width : byte;
                Vert, Horiz,              { box's wall characters }
                UpLeft, UpRight,          { corner characters }
                LowLeft, LowRight,
                BubbleChar : char;        { bubbles to come from Owl's head }
                LinesToClear : byte;      { number of lines to erase }
              end;
const
  BlinkOwl = True;
  OwlX = 55;
  OwlY = 13;
  OwlBubble : BubbleRec =
               (X : 1; Y   : 1;             { box's dimensions      }
                Height     : 11;            { OwlY - 2              }
                Width      : 56;            { Succ(OwlX)            }
                Vert       : '';           { box's wall characters }
                Horiz      : '';
                UpLeft     : '';           { corner characters     }
                UpRight    : '';
                LowLeft    : '';
                LowRight   : '';
                BubbleChar : 'o';
                LinesToClear : 0);          { number of lines to erase }
var
  Root  : AnimalPtr;
  Rec   : StringRec;
  F     : text;
  Score : ScoreRec;

procedure Abort(Msg : MaxString);
{ Abort the program, display an error message. }
begin
  GotoXY(1, MaxRows);
  Textcolor(15);
  Write(Msg, '.  Program aborted');
  Halt;
end; { Abort }

procedure CreateNode(var P : AnimalPtr;
                Beast : NameStr;
                Query : QuestionStr;
            var Count : integer);    { either Score.Added or score.Total }
{ Allocate a new record from the heap; halt the program if
  there is not enough memory.
}
begin
  if Abs(MaxAvail) < SizeOf(P^) then
    Abort('Out of memory');
  Count := Succ(Count);
  New(P);
  with P^ do
  begin
    Name := Beast;
    Question := Query;
    Yes := nil;
    No := nil;
  end;
end; { CreateNode }

procedure FillTree(var Root : AnimalPtr);
{ This procedure checks for the data file called DataFile.
  If the file exists, it gets the first animal from the file and
  then calls LoadTree to build and load the tree of known animals.
  If the file does not exist, it creates a tree with one animal.
}

procedure GetRec(var Rec : StringRec);
{ Reads a "record" from the disk file. }
begin
  with Rec do
  begin
    Readln(F, Name);
    StringP := Name;               { assume it's a NIL stmt }
    if Name = 'NIL' then
      Exit;     { Name & question will be ignored }
    Readln(F, Question);
    StringP := '';
  end; { with }
end; { GetRec }

procedure LoadTree(var Rec : StringRec; var Animal : AnimalPtr);
{ Loads the tree from a disk file at program start up. }
begin
  if Rec.StringP = 'END' then
    Exit;        { EOF }
  if Rec.StringP <> 'NIL' then
  begin
    CreateNode(Animal, Rec.Name, Rec.Question, Score.Total);
    GetRec(Rec);
    LoadTree(Rec, Animal^.Yes);
    LoadTree(Rec, Animal^.No);
  end
  else
    GetRec(Rec);
end; { LoadTree }

function OpenFile(var F : text; FileName : MaxString) : boolean;
{ TRUE if can successfully open FILENAME, else FALSE }
begin
  Assign(F, FileName);
  {$I-}
  Reset(F);
  {$I+}
  OpenFile := IOresult = 0;
end; { OpenFile }

begin { FillTree }
  FillChar(Score, SizeOf(Score), #0);        { init score variables }
  if OpenFile(F, DataFile) then              { found the disk file  }
  begin
    textcolor(13);
    Writeln('Loading from disk...');
    GetRec(Rec);
    LoadTree(Rec, Root);
  end
  else                              { empty tree: create root entry }
    CreateNode(Root, 'snake', 'It''s cold blooded w/scaly skin', Score.Total);
  TextBackground(1);
  ClrScr;
end; { FillTree }

procedure SaveTree(var Root : AnimalPtr);
{ This procedure saves the tree (pointed to by "Root") to a disk
  file.  Note that file was "opened" by FillTree at program start up.
}

procedure DumpTree(Animal : AnimalPtr);
{ Recursive procedure that dumps the tree to a text file.
  The global file variable "f" is already "opened" by FillTree
  and SaveTree.  DumpTree uses a pre-order traversal of the tree.
}
begin
  if Animal <> nil then
  begin
    Writeln(F, Animal^.Name);
    Writeln(F, Animal^.Question);
    DumpTree(Animal^.Yes);
    DumpTree(Animal^.No);
  end
  else
    Writeln(F, 'NIL');     { indicates a nil pointer in the text file }
end; { DumpTree }

begin { SaveTree }
  GotoXY(1, MaxRows);
  if Score.Added <> 0 then            { any changes made to the tree? }
  begin
    textcolor(lightmagenta);
    Write('Saving animals...');
    Rewrite(F);
    DumpTree(Root);
    Writeln(F, 'END');       { indicates END of tree data in text file }
    Close(F);
  end;
  textmode(lastmode);
  Write(#13, 'Bye!!!');  { go to column #1, display msg }
end; { SaveTree }

function AorAn(S : MaxString) : MaxString;
{ This function is given a string; if the string begins with a vowel,
  it returns the article "a", a space and the original word, otherwise
  it returns the article "an", a space and the original word.
}
const
  Vowels : set of char = ['A', 'E', 'I', 'O', 'U'];
begin
  if UpCase(S[1]) in Vowels then
    AorAn := 'an ' + S
  else
    AorAn := 'a ' + S;
end; { AorAn }

{ =================  Cartoon routines =============================== }
procedure DrawBubble(Bubble : BubbleRec);
{ Draws a bubble-shaped window.  On horizontal lines, draws a space
  every other character to make the box "softer" and more like a bubble.
}
var B : byte;
begin
  TextColor(LightCyan);
  with Bubble do
  begin
    GotoXY(Succ(X), succ(Y));              { upper left corner }
    for B := 1 to Width - 2 do       { top border }
      if Odd(B) then
        Write('-')
      else
        Write(Horiz);
    GotoXY(X, Y + Pred(Height));     { lower left corner }
    Write(LowLeft);
    for B := 1 to Width - 2 do       { bottom border }
      if Odd(B) then
        Write('-')
      else
        Write(Horiz);
    Write(LowRight);                 { lower right corner }
    for B := 1 to Height - 2 do      { bottom border }
    begin
      GotoXY(X, Y + B);              { left border }
      Write(Vert);
      GotoXY(X + Pred(Width), Y + B);{ right border }
      Write(Vert);
    end; { for }
    GotoXY(X, Succ(Y));              { upper left corner }
    Write(UpLeft);
    GotoXY(X + Pred(Width), Succ(Y));  { upper left corner }
    Write(UpRight);
    GotoXY(X + Width, Y + Height);
    Write(BubbleChar);
    GotoXY(Succ(X + Width), Succ(Y + Height));
    Write(BubbleChar);
  end; { with }
end; { DrawBubble }

procedure ClrBubble(var Bubble : BubbleRec);
{ Clears the "bubble" window of text. }
begin
  TextColor(LightCyan);
  window(2,3,55,10);
  clrscr;
  window(1,1,80,25);
end;

type
  Tasks = (EntireOwl, EyesOpen, EyesClosed);
procedure DrawOwl(Task : Tasks; X, Y : byte);
{ A cartooning routine that draws an owl.  It can be used
  to "blink" the eyes by calling it with various TASK parameters.
}
var I : integer;
begin
  TextColor(brown);
  case Task of
    EntireOwl : begin
                  ClrScr;
                  TextColor(Brown);
                  for I := 0 to 8 do
                  begin
                    GotoXY(X, Y + I);
                    case I of
                      0 : Writeln('     /\,,,/\');
                      1 : Writeln('    {(O) (O)}');
                      2 : Writeln('     |  V  |');
                      3 : Writeln('   //|||||||\\');
                      4 : Writeln('////|||||||||\\\\');
                      5 : Writeln('///  |||||||  \\\');
                      6 : Writeln('//   ||/ \||   \\');
                      7 : Writeln('     `.."..'' ');
                      8 : Writeln('======`''=`''==============');
                    end; { case }
                  end; { for }
                end;
    EyesOpen  : begin
                  GotoXY(X + 6, Succ(Y));
                  TextColor(Brown);
                  Write('O) (O');
               end;
    EyesClosed: begin
                  GotoXY(X + 6, Succ(Y));
                  TextColor(Brown);
                  Write('-) (-');
                end;
  end; { case }
end; { DrawOwl }

procedure BlinkEyes(CurX, CurY : byte);
{ A cartooning routine to blink the owl's eyes.  It is called by
  the GetCh routine; it blinks the owl's eyes until a key is typed,
  then it makes sure the owl's eyes are "open" and then exits.
  It is passed the current x,y cursor coordinates, blinks the eys,
  and then returns to the prev. cursor position upon exit.
}

var
  BlinkCount : integer;
  OpenDuration,          { keep eyes open @ 200/hundredths of a second }
  ClosedDuration:byte;   { keep eyes closed @ 120/hundredths of a second }
begin
  BlinkCount := 0;
  repeat
    Openduration:=random(200)+50;
    ClosedDuration:=random(100)+50;
    BlinkCount := Succ(BlinkCount);
    Delay(10);
    if BlinkCount mod OpenDuration = 0 then
    begin
      if KeyPressed then
        Exit;
      DrawOwl(EyesClosed, OwlX, OwlY);
      GotoXY(CurX, CurY);              { return to former cursor position }
      repeat
        BlinkCount := Succ(BlinkCount);
        Delay(10);
      until KeyPressed or (BlinkCount mod ClosedDuration = 0);
      DrawOwl(EyesOpen, OwlX, OwlY);
      GotoXY(CurX, CurY);              { return to former cursor position }
      BlinkCount := 0;
    end;
  until KeyPressed;
end; { BlinkEyes }

{ ============== beginning of editing routines ============== }
type
  CharSet = set of char;
procedure GetCh(X, Y : byte;         { x,y coordinates }
              var Ch : char;         { character to read }
            LegalSet : CharSet;      { legal input characters }
            BlinkOwl : boolean);     { whether to blink the owl's eyes }
{ Read a character from the keyboard.  Keep reading characters until:
    o a ^C is typed (abort the program)
    o a legal character is typed (specified by LegalSet parameter)
  In addition, if we are blinking the owl's eyes, call the cartooning
  routine BlinkEyes.
}
begin
  repeat
    if BlinkOwl then
      BlinkEyes(X, Y);
    ch:=readkey;
    if (Ch = #27) and KeyPressed then
    begin
      ch:=readkey;
      Ch := Chr(Ord(Ch) or $80);               { turn on high bit }
    end;
    if Ch in [^C] then
      Abort('** Whoops!  I was INTERRUPTED! STOPPED!!');
  until UpCase(Ch) in LegalSet;
end; { GetCh }

const
  TermSet : CharSet = [ESC, CR];  { set of terminating chars. }
function GetString(var St : MaxString;   { string to edit }
                   MaxLen : byte;        { max length of str }
                     X, Y : byte         { starting row,col }
                        ) : char;        { returns terminating char }

{ GetString edits a string.  It is given a string variable (St), the
  maximum allowable length of the string (MaxLen), and the x,y coor-
  dinates of the first character in the string.  It uses GetCh to get
  characters from the keyboard and the global set constant TermSet to
  determine when to stop editing the string.  It returns the terminating
  character as the function value.  Input characters are highlighted
  (HighVideo), a stream of periods ('.') shows the user how many input
  characters are allowed.
}

procedure AddChar(Ch : char; var S : MaxString);
{ Add a character to the end of the string. }
begin
  TextColor(15);
  Write(Ch);                    { display new char }
  TextColor(LightCyan);
  S := S + Ch;                  { add to string }
end; { AddChar }

procedure DelChar(var S : MaxString);
{ Delete a character from the end of the string. }
begin
  GotoXY(Pred(X + Length(S)), Y);  { to last char }
  TextColor(LightCyan);
  Write('.');                     { overwrite it }
  Delete(S, Length(S), 1);        { remove last char }
  GotoXY(X + Length(S), Y);       { to end of string }
end; { DelChar }

var
  Ch : char;
  B : byte;
begin { GetString }
  GotoXY(X, Y);                     { use periods to show maximum len }
  TextColor(LightCyan);
  for B := 1 to MaxLen do       { fill out display field with periods }
    Write('.');
  GotoXY(X, Y);
  TextColor(15);
  Write(St);
  GetString := Null;             { Assume no terminating action taken }
  repeat
    GetCh(X + Length(St), Y, Ch, [#0..#255], BlinkOwl); { to end of string }
    if not (Ch in TermSet) then
    begin
      case Ch of
        ' '..#127 : if Length(St) < MaxLen then  { still room for more }
                      AddChar(Ch, St);
        BS, DEL   : if Length(St) > 0 then
                      DelChar(St);
        ^X,^U,^A  : while Length(St) > 0 do              { erase string }
                      DelChar(St);
        else;                      { case else: ignore other characters }
      end; { case }
    end;
  until Ch in TermSet;
  GetString := Ch;
end; { GetString }
{ ===================== end of editing routines ===================== }

procedure GuessAnimal(var Animal : AnimalPtr;
                          var Ch : char);            { returns answer }
{ Recursive routine that "guesses" the user's animal.  It is given
  a pointer to an "Animal."  It asks the user the question associated
  with this "animal" (done by ThisQuestion); if the user answers yes
  to the question, it "guesses" the animal's name (done by ThisAnimal);
  if the animal was "guessed" correctly, the procedure boasts about its
  accomplishment, the unwinds the stack and returns to the calling routine.

  If it did not guess the animal, it calls itself and passes the appropriate
  pointer associate with each animal (a YES description follows the YES
  path, a NO to description follows the NO path...).
}

function ThisQuestion(Animal : AnimalPtr;
                      var Ch : char          { returns the answer }
                           ) : boolean;
{ Ask the user a question using the description associated with this
  animal; return TRUE if the user YES, otherwise return FALSE;  return
  the answer in the char variable parameter.
}
const
  Qx = 3;       { x coordinate of where question will be asked }
  Qy = 3;       { y coordinate of where question will be asked }
begin { ThisQuestion }
  ClrBubble(OwlBubble);
  GotoXY(Qx, Qy);
  TextColor(15);
  Write(Animal^.Question, '? ');                        { Ask the question }
  TextColor(LightCyan);
  Write('[Y,N] ');
  TextColor(15);
  GetCh(Qx + Length(Animal^.Question) + 8, Qy,           { x,y coordinates }
        Ch,                                              { read this char  }
        ['Y', 'N', #27],                                 { set of answers  }
        BlinkOwl);                                       { do cartooning   }
  OwlBubble.LinesToClear := Qy;             { set flag: dirtied the window }
  ThisQuestion := UpCase(Ch) = 'Y';
end; { ThisQuestion }

function ThisAnimal(Animal : AnimalPtr;
                    var Ch : char          { returns the answer }
                         ) : boolean;
{ Guess the animal's name using the animal associated with this record:
  return TRUE if the user YES, otherwise return FALSE;  return
  the answer in the char variable parameter.
}
const
  Ax = 3; { x coordinate of where question will be asked }
  Ay = 4; { y coordinate of where question will be asked }
var S : MaxString;
begin
  GotoXY(Ax, Ay);
  TextColor(15);
  S := 'Is it ' + AorAn(Animal^.Name) + '? ';  { the question to ask }
  Write(S);
  TextColor(LightCyan);
  Write('[Y,N] ');
  TextColor(15);
  GetCh(Ax + Length(S) + 6, Ay,                     { x,y coordinates }
        Ch,                                         { char to read    }
        ['Y','N',#27],                              { legal answers   }
        BlinkOwl);                                  { do cartooning   }
  OwlBubble.LinesToClear := Ay;        { set flag: dirtied the window }
  ThisAnimal := UpCase(Ch) = 'Y';
end; { ThisAnimal }

procedure AddAnimal(var Animal : AnimalPtr;
                        var Ch : char);          { returns value }
{ This routine adds another animal to the tree.  It asks the user
  for the animal's name; it asks for a brief description of the
  animal (which it will later read as if it was a question).  It
  then adds the animal to the tree.  The terminating character from
  the editing routine is returned in the char parameter.
}
const
  Ax = 3;              { x coordinate of where animal name is entered }
  Ay = 6;              { y coordinate of where animal name is entered }
  Qx = 3;              { x coordinate of where description is entered }
  Qy = 9;              { y coordinate of where description is entered }
var
  Ques : QuestionStr;                           { description entered }
  Beast : NameStr;                              { name entered        }
  loop : integer;                               { silly sound         }
begin
  GotoXY(Ax, Ay);
  TextColor(15);
  with OwlBubble do
    LinesToClear := Pred(Height);   { set flag: we dirtied the bubble }

  for loop:=10000 downto 1 do sound(loop);
  nosound;
  Write('Ok, ok, I give up.  What''s your animal''s name:');
  Beast := '';
  Ch := GetString(Beast, Pred(SizeOf(Beast)), Ax, Succ(Ay));  { get name }
  if Ch = ESC then
    Exit;                                { ESC: Don't add }
  GotoXY(Qx, Qy);
  Write('What''s ', AorAn(Beast), ' like:');
  Ques := '';
  Ch := GetString(Ques, Pred(SizeOf(Ques)), Qx, Succ(Qy));  { get descrip. }
  if (Ques <> '') and              { legal name & description? Add to tree }
     (Beast <> '') and
     (Ch <> ESC) then
    CreateNode(Animal, Beast, Ques, Score.Added);
end; { AddAnimal }

procedure Boast;
{ Boast after having guessed the animal correctly. }
var
   loop:integer;                                          { silly sound }

begin
  Write('Yes');
  with OwlBubble do
    GotoXY(X + 2, Y + Height - 2);  { last row }
  TextColor(15);
  Write('-- AHA! I knew it was ', AorAn(Animal^.Name), '!');
  Score.Guessed := Succ(Score.Guessed);
  for loop:=1 to 15000 do
      sound(loop);
  nosound;
  Delay(2000);
  with OwlBubble do
    LinesToClear := Pred(Height);  { we dirtied the bubble }
end; { Boast }

begin { GuessAnimal }
  if Ch = ESC then
    Exit;                           { return to main menu }
  if Animal <> nil then                          { more animals to guess }
  begin
    if ThisQuestion(Animal, Ch) then
    begin
      Write('Yes');
      if ThisAnimal(Animal, Ch) then
        Boast                                        { confirm our guess }
      else
        GuessAnimal(Animal^.Yes, Ch)   { right description, wrong animal }
    end { if }                         { follow YES path                 }
    else
      GuessAnimal(Animal^.No, Ch)      { wrong description, wrong animal }
  end { if }                           { follow NO path                  }
  else                    { No more animals on this path: add the animal }
  begin
    Write('No');
    AddAnimal(Animal, Ch);
  end; { else }
end; { GuessAnimal }

function WantToGuess(var Ch : char) : boolean;
{ Display scoring information, check whether user wants to
  play another round.
}
var
  Col, Line : byte;

procedure GoLine(X : byte; var Y : byte);
{ Go to line Y, increment Y }
begin
  GotoXY(X, Y);
  Y := Succ(Y);
end; { GoLine }

procedure ShowBubbleMsg;
{ Fills the bubble window with text that explains the game. }
begin
  with OwlBubble do
  begin
    Write('':Pred((Width - 18 - Col) div 2), 'THINK OF AN ANIMAL');
    TextColor(LightCyan);
    Line := Succ(Line);
    GoLine(Col, Line);
    with Score do
    begin
      case Total + Added of
        1 : Write('I only know one animal, but I can guess or learn');
        else
          Write('I know ', Total + Added,
                ' different animals and can guess');
      end; { case }
    end; { with }
  end; { with }
  GoLine(Col, Line);
  Write('the ones you think of.  Think of an animal');
  GoLine(Col, Line);
  Write('if you want to play.  ');
  TextColor(15);
end; { ShowBubbleMsg }

begin { WantToGuess }
  TextColor(LightCyan);                                 { show scoring info }
  GotoXY(67, Pred(MaxRows));
  Write('Guessed', Score.Guessed:6);
  GotoXY(67, MaxRows);
  Write('Added  ', Score.Added:6);
  with OwlBubble do
  begin
    Line := Y + 2;
    Col := X + 3;
    GoLine(Col, Line);
    TextColor(15);
    ShowBubbleMsg;                                       { paint "menu" }
    GotoXY(Col, Succ(Line));
    Write('Ready? ');
    GetCh(Col + 7, Succ(Line),                    { x,y coordinates }
          Ch,                                     { char to read    }
          ['Y','N', #27],                         { legal answers   }
          not BlinkOwl);                          { don't blink eyes }
    if UpCase(Ch) <> 'Y' then
      Write('No');
    LinesToClear := Pred(Height);     { set flag: we dirtied the bubble }
  end;
  WantToGuess := UpCase(Ch) = 'Y';
end; { WantToGuess }

var
  Ch : char;         { used by entry routines: returns terminating char }
  Quit : boolean;                               { loop control variable }
begin { program body }
  CheckBreak:=False;
  FillTree(Root);                      { load from disk or build if new }
  DrawOwl(EntireOwl, OwlX, OwlY);                        { paint screen }
  DrawBubble(OwlBubble);
  repeat
    Quit := not WantToGuess(Ch);
    if not Quit then
    begin
      GuessAnimal(Root, Ch);                         { guess the animal }
      ClrBubble(OwlBubble)                           { clear the window }
    end;
  until Quit;
  SaveTree(Root);                            { save animal data to disk }
end.
